/** ***
 *  RCS_EE module contains EEPROM functions
 *  Stores the program ID and version number,
 *  Stores the selected start sequence ID and number of flights
 *    to provide default at next power up.
 *  Copyright (c) Chris Hearn 2017
 */

// include external libraries:
#include <EEPROM.h> // store selected sequence for next power up.
#include <Arduino.h>
#include <RCS_EE.h>

namespace RCS {

  

/* *** Defined storage locations in EEPROM ***/
/*  EEPROM address 0,1 is a word pointer to the first unused address (0..1023) will always be higher than 8:
 *    Addr 2..5 stores a 4 byte ID for this prog (RCSv) 
 *    Addr 6 = 1 byte with major/minor version numbers
 *    Addr 7 = 1 byte for start sequence number
 *    Addr 8 = 1 byte for numFlights
 *    Followed by any other data (future)
 */

  
#define loc_pointer 0     // 2 byte address
#define loc_header  2     // stores "RCSv" identifies our program HAS stored data already
const char strKey[4] = {'R','C','S','v'}; // our program ID
  
  bool _debug =false;       // if true, will output messages to Serial.
  
    // constructor
  EE::EE(bool debug) {
    _debug = debug;
    
  }

void EE::writeHeader(byte progVersion) {
  EEPROM.update(loc_pointer,9); // int lsb of pointer (to next free address - not used at present)
  EEPROM.update(loc_pointer+1,0); // int msb of pointer
  // write 'RCSv'
  EEPROM.update(loc_header+0,strKey[0]);
  EEPROM.update(loc_header+1,strKey[1]);
  EEPROM.update(loc_header+2,strKey[2]);
  EEPROM.update(loc_header+3,strKey[3]);

  writeVersion(progVersion);
}

  
byte EE::readByte(int epAddress)
{
  return EEPROM.read(loc_header+epAddress);
}
  

byte EE::readVersion()
{
  return EEPROM.read(loc_header+4);
}


void EE::writeVersion(byte vNum)
{
  EEPROM.update(loc_header+4,vNum);  
}


bool EE::isCurrentVersion(byte progVersion)
{
  return ( readVersion() == progVersion);
}


byte EE::readSequenceNumber()
{
  return EEPROM.read(loc_header+5);
}


void EE::writeSequenceNumber(byte seqNo)
{
  if (_debug) { Serial.print(F("Storing start sequence ID: ")); Serial.println(seqNo); }
  EEPROM.update(loc_header+5,seqNo);
}


byte EE::readNumberOfFlights()
{
  return EEPROM.read(loc_header+6);
}


void EE::writeNumberOfFlights(byte flights)
{
  if (_debug) { Serial.print(F("Storing No. Flights = ")); Serial.println(flights); }
  EEPROM.update(loc_header+6,flights);
}


void EE::setSoftwareVersion(byte progVersion) {
  if ( isCurrentVersion(progVersion) == false)
  {
    writeHeader(progVersion);
    storeStartSettings(0,1); // default to 1st startSequence and 1 flight
  }
}


void EE::storeStartSettings(byte seqID, byte nFlights) {
  writeSequenceNumber(seqID);  // save/update to EEPROM
  writeNumberOfFlights(nFlights); // ditto
}

}
