// Arduino RCS Timer Library v1.0.0
// Modified to combine RBD_Timer and RBD_Button and remove unwanted methods
// Thanks to Alex Taujenis
// https://github.com/alextaujenis/RBD_Timer
// Copyright 2016 Alex Taujenis
// MIT License

#include <Arduino.h>
#include <RCS.h>




namespace RCS {
  Timer::Timer() {}

  Timer::Timer(unsigned long value) {
    setTimeout(value);
  }

    // private
  
  void Timer::_updateState() {
    if(_state == ACTIVE && getValue() >= _timeout) {
      _state = EXPIRED;
    }
  }

    // public
  
  unsigned long Timer::getValue() {
    return millis() - _waypoint;
  }
  
  unsigned long Timer::getTimeout() {
    return _timeout;
  }
  
  void Timer::setTimeout(unsigned long value) {
    _timeout = (value > 0) ? value : 1;
  }


  void Timer::restart() {
    _waypoint         = millis();
    _state            = ACTIVE;
    _has_been_active  = false;
    _has_been_expired = false;
  }

  void Timer::stop() {
    _state = STOPPED;
  }

  bool Timer::isActive() {
    _updateState();
    return _state == ACTIVE;
  }

  bool Timer::isExpired() {
    _updateState();
    return _state == EXPIRED;
  }

  bool Timer::isStopped() {
    return _state == STOPPED;
  }
  
  bool Timer::onRestart() {
    if(isExpired()) {
      restart();
      return true;
    }
    return false;
  }

  bool Timer::onActive() {
    if(!_has_been_active && isActive()) {
      return _has_been_active = true;
    }
    return false;
  }

  bool Timer::onExpired() {
    if(!_has_been_expired && isExpired()) {
      return _has_been_expired = true;
    }
    return false;
  }



    // input pullup enabled by default
  Button::Button(int pin)
  : _pressed_debounce(), _released_debounce() {
    _pin = pin;
    _inputPullup();
    setDebounceTimeout(_debounce_timeout);
  }

    // overloaded constructor to disable input pullup
  Button::Button(int pin, bool input_pullup)
  : _pressed_debounce(), _released_debounce() {
    _pin = pin;
    if(input_pullup) {_inputPullup();}
    else {_disableInputPullup();}
    setDebounceTimeout(_debounce_timeout);
  }

  void Button::setDebounceTimeout(unsigned long value) {
    _debounce_timeout = value;
    _pressed_debounce.setTimeout(_debounce_timeout);
    _released_debounce.setTimeout(_debounce_timeout);
  }

  bool Button::isPressed() {
    _temp_reading = digitalRead(_pin);
    if(_invert) {return !_temp_reading;}
    else {return _temp_reading;}
  }

  bool Button::isReleased() {
    return !isPressed();
  }

  bool Button::onPressed() {
    if(isPressed()) {
      _pressed_debounce.restart();
      if(!_has_been_pressed) {
        return _has_been_pressed = true;
      }
      return false;
    }
      // else
    if(_pressed_debounce.isExpired()) {
      return _has_been_pressed = false;
    }
    else {
      return false;
    }
  }

  bool Button::onReleased() {
    if(isReleased()) {
      _released_debounce.restart();
      if(!_has_been_released) {
        return _has_been_released = true;
      }
      return false;
    }
      // else isPressed()
    if(_released_debounce.isExpired()) {
      return _has_been_released = false;
    }
    else {
      return false;
    }
  }

  void Button::invertReading() {
    _invert = !_invert;
  }


    // private

  void Button::_inputPullup() {
    pinMode(_pin, INPUT_PULLUP);
  }

  void Button::_disableInputPullup() {
    pinMode(_pin, INPUT);
  }
}
