
/* A portable sailing RaceBox control system
 *
 *  Author: Chris Hearn
 *  Version: 1.1
 *  Date: 27 June 2017
 *  
 *  
 *  History
 *    v1.1 27-June-2017
 *      Now easily change relay logic to work with active HIGH or LOW relay shield
 *        Usage: Set boolean activeRelayState = HIGH/LOW below, as required.
 *    v1.0 23-June-2017
 *      Initial Release.
 */

 /*      
  License:
  This is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This code is provided in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  External Libraries:
  Thanks to the Authors: 
  https://github.com/PaulStoffregen/FlexiTimer2 for flexiTimer2 library
  and
  https://github.com/alextaujenis/ for RBD Button and Timer classes.
  
 */  

 
 /*  Features:
 *  Selectable starting countdown sequences.
 *  Outputs for both sound (Horn) and Lamp (relay) signals.
 *  LCD display shows countdown/elapsed time, which flags to raise/lower etc.
 *  Warning buzzer before actions to prompt the P.R.O
 *  Buttons for Configuration, Start, Horn and Individual Recall/Class Recall/Shorten Course
 *  Auto horn signals, plus Manual Horn button can be used at any time
 */

/*
 * INSTALLATION
 * Download as 4 zip files:
 *  1. RaceBox.zip - the main program and supporting files. Unzip into your sketchbook folder
 * 
 * Libraries: Add to your Sketchbook/libraries folder.
 *  With the Arduino IDE, select menu Sketch/Include Library and "Add .ZIP library"
 * 2. flexiTimer2.zip
 * 3. RCS_zip
 * 4. RCS_EE.zip
 * 
 * Compile the sketch and then upload to Arduino UNO in the stanard way.
 */
   
/*  HARDWARE:
 *  Arduino Uno or compatible microcontroller
 *  Relay Shield - for 4 isolated relay outputs to lamps
 *  LCD to give information.
 *  4 momentary buttons for action inputs
 *  5 LED's for 'Run', 'Horn' and lamp status mimics
 *  
 *  PIN ASSIGNMENTS:
 *  Pin  TYPE           Assignment 
 *   2  INPUT_PULLUP    Start Button
 *   3  OUTPUT          Flashing LED when running
 *   4  OUTPUT          Relay 4: Class Lamp output 
 *   5  OUTPUT          Relay 3: Prep Lamp output
 *   6  OUTPUT          Relay 2: Warn/Recall/Shorten Course output
 *   7  OUTPUT          Relay 1: HORN output
 *   8  INPUT_PULLUP    Recall Button - Recall/ShortenCourse and "configure start sequence"
 *   9  INPUT_PULLUP    Horn Button - Manual Horn
 *   10 OUTPUT          Piezo sounder
 *   11 OUTPUT          LCD
 *   12 OUTPUT          LCD
 *   14 GND
 *   A0 OUTPUT          LCD (Analogue outputs for LCD data)
 *   A1 OUTPUT          LCD (Analogue outputs for LCD data)
 *   A2 OUTPUT          LCD (Analogue outputs for LCD data)
 *   A3 OUTPUT          LCD (Analogue outputs for LCD data)
 *   A4   N.C             
 *   A5 INPUT_PULLUP    (Used as digital input) Set Button
 *   
*/



/**** CODE - Herewith lie demons - take care! ****/

//  Put the version number below. Change it when code here OR in included .cpp files changes
const byte versionNumber = 0x11; // majorVersion * 16 + minorVersion; so v0.2 = 0x02, v1.2 = 0x12; etc

/*
 * Debugging variables
 */
const bool debug = false; // for debugging - (also enables Serial.print(msg) in the giveMsg() method )

// for "real time" set speedUp = 1 
const int speedUp = 1; // But we can go 2 or 3 times faster whilst debugging!


/* macro to get number of elements in an array
 */
#define arr_len( x )  ( sizeof( x ) / sizeof( *x ) )


/* 
 *  include external libraries:
 */
#include <avr/pgmspace.h>   // for read/write to PROGMEM -e.g. read a Start sequence array into RAM using pgm_read_word_near()

#define haveLCD true    // Do we have an LCD for messages? Recommend we do!

#ifdef haveLCD
  #include <LiquidCrystal.h>  // For Setup, race messages and debugging
  // LCD is 16x2
  LiquidCrystal lcd(11, 12, A3, A2, A1, A0); // initialize the library with the pins. Pins A0-A3 for data
#endif

// NOTE: these should be in the sketchbook/Libraries folder
#include "FlexiTimer2.h"    // For the main (accurate) interrupt-driven Race timer.
#include "RCS.h"        // Global defines and static global variables, plus HornTimer and Button class methods
#include "RCS_EE.h"  // EEPROM methods


/* Race constants
 *
 */
  
  // PINOUT Definitions
  
#define startBtnPin   2
#define stateLEDPin   3
#define hornBtnPin    9
#define recallBtnPin  8
#define buzzerPin     10
#define setBtnPin     A5 // btn was previously connected to RESET, now used as digital INPUT.

/*  *** RELAYS ***
 * Assign the active state and relay pins to match the hardware used!
 * 
 *  The Seed Studio Relay Shield v3.0 provides 4 relays controlled by pins 4-7 for Relays 4,3,2,1 respectively
 * see http://wiki.seeed.cc/Relay_Shield_v3/
 *  The relays are turned on with a HIGH on the control pin. i.e "Active HIGH" 
 */
const boolean relayActiveState = HIGH;

  // Relay pin definitions
  
#define classRelayPin   4 // RELAY #4
#define prepRelayPin    5 // RELAY #3
#define warnRelayPin    6 // RELAY #2
#define recallRelayPin  6 // same relay used for both Warn (before the start) and Recall (after the start) functions
#define hornRelayPin    7 // RELAY #1


/* HornTimer and Store objects
 *  Initialise
 */
RCS::Timer hornTimer(int(1500/speedUp)); // init object in global space, set Horn timeout to 1.5 secs (single Hoot) for LIVE
  
RCS::EE Store(debug); // init object instance to read/write to EEPROM

/*
 * Define key race array constants
 */
  // CAUTION *** numSignals MUST equal the size of start arrays below ***
#define numSignals 9      // number of elements in astartTimes[] and the aTeamStartTimes[] - must be same size! (set any elements not wanted to NOTUSED)
#define NOTUSED 99        // enables skipping a sequence element (e.g. NO RaceWarning wanted)


/*
 * STATIC GLOBALS & const arrays. They are stored in PROGMEM, which has the space!
 */
static byte aSTlength = numSignals;  // default, to match the length of start arrays below
static byte tZeroIndex = 4; // default, the array index that holds zero (i.e. GO)  actual value calculated at startup.

const int enableShortCourseAfterSecs = 600; // enable button for 'Shorten Course' this # secs after last flight started

/* message strings for the LCD display */
const char *aStartNames[9]  = {"RACE WARNING", "CLASS FLAG UP","PREP FLAG UP ","PREP FLAG DOWN","START-CLASS DOWN",
  "NEXT FLIGHT","PREP FLAG UP ","PREP FLAG DOWN","START-CLASS DOWN"};


/*** THE START SEQUENCE names 
 * NOTE: Team Races are indiated by prefix of "T-" and provide extra countdown sound signals. 
 * See details below, aTeamStartTimes[numSignals] and aTeamSounds[numSignals]
 */
const char *aSequenceNames[12]  = {"541GO", "6-541GO", "10-541GO", "321GO", "4-321GO", "5-321GO", "10-321GO", "631GO", "9-631GO", "963GO", "T-321GO", "T-541GO"};

/*** START SEQUENCES:
 * Are loaded into PROGMEM (non volatile), 
 *  Timings are in seconds w.r.t. Flight 1 GO
 *  The repeat sequence comes AFTER T0  (for each following flight start)
 * 
 * If used 'RACE WARINING' is not actually part of ISAF standards, but is used by many clubs. 
 * Assign NOTUSED to any signal that should be ignored (except T0 and last array item).
 * Sequence actions are: RACE WARNING, CLASS FLAG UP, PREP FLAG UP, PREP FLAG DOWN, GO =  CLASS FLAG DOWN
 * 
 * TEAM RACING - Represented by "T-" in the sequence name.
 * Gives extra single hoots in the last minute, at -30, -20, -10, and -5,-4,-3,-2,-1 seconds
 * ID = 11 is T-321GO TEAM RACING  Lights as ID = 3, with extra sounds
 * ID = 12 is T-541GO TEAM RACING Lights as ID = 0,  ditto


*/
const PROGMEM int16_t aStartSeqs[][numSignals] = {
  { NOTUSED, -300, -240, -60, 0,   15, 60, 240, 300 },      // 541Go.    // ISAF RULE 26 ( 2 flights )
  { -360,    -300, -240, -60, 0,   15, 60, 240, 300 },      // 6-541Go   // ISAF RULE 26 with 6 minute RACE WARNING
  { -600,    -300, -240, -60, 0,   15, 60, 240, 300 },      // 10-541Go  // ISAF RULE 26 with 10 minute RACE WARNING
  { NOTUSED, -180, -120, -60, 0,   15, 60, 120, 180 },      // 321Go
  { -240,    -180, -120, -60, 0,   15, 60, 120, 180 },      // 4-321Go
  { -300,    -180, -120, -60, 0,   15, 60, 120, 180 },      // 5-321Go
  { -600,    -180, -120, -60, 0,   15, 60, 120, 180 },      // 10-321Go
  { NOTUSED, -360, -180, -60, 0,   15, 180, 300, 360 },     // 631Go     // std sequence, but non-std times, 6minutes flights
  { -540,    -300, -180, -60, 0,   15, 60, 120, 180 },      // 9-631Go   // std sequence, but non-std times, 3 minute flights
  { -540, -360, -180, NOTUSED, 0, +15, +16, NOTUSED,+180 }, // 963Go // non-ISAF, 3 minute flights
  { NOTUSED, -180, -120, -60, 0,   15, 60, 120, 180 },      // T-321Go   // TEAM RACING - has extra sound signals
  { NOTUSED, -300, -240, -60, 0,   15, 60, 240, 300 }       // T-541Go
};

/* Additional Command messages */
const char* aRaceNamesx[5] = {"RECALL:INDIV.","RECALL:FLIGHT","SHORTEN COURSE"};



/* SOUNDS:
 * More flexible, we define the possible sound sequences in this format:
 * {number of sounds,units_length, pause_length) in 1/10secs. Pause_length is ignored for single sound signals
 * e.g. Race Warning signal. 5 hoots of 0.5s with 0.5s gap is { 5,5,5 }
 */
const byte aSoundDefs[][3] = { 
    {0,0,0},    // no sound
    {5,5,5},    // 5 x 0.5sec hoots
    {1,15,0},   // 1 x 1.5sec hoot
    {1,25,0},   // 1 x 2.5sec hoot
    {2,15,5},   // 2 x 1.5sec hoots
    {3,15,5},   // 3 x 1.5sec hoots
    };


/* Key race info 
 * These MUST be in RAM 
 * During the race we update values >=0 after T0 (add flight interval to each) as each flight starts.
 */
static int flightInterval = 180; // default to 3 minutes.
  
int  aStartTimes[numSignals] = { -300, -180, -120, -60,  0, 15, 60, 120, 180 };  // default to 5-321Go, but will be overwritten!

 
/* SOUND TIMING
 * For a start Sequence, define the times and the sound index into aSoundDefs[]
 * e.g.  
 *   int aStartTimes[] = { NOTUSED, -300, -240, -60,  0,  15, 60, 240, 300 }; 
 *   byte aSounds[] =    {    1,       2,    2,   3,  2,   0,  2,   3,   2 };
 *   this will give {RaceWarning-nothing, HOOT (Class UP), HOOT(Prep UP), LONG_HOOT (Prep Down,1 minute), HOOT (at T0), 
 *      Nothing at +15, HOOT at +60, LONG_HOOT at +240, HOOT at +300}
 * 
 */
byte aSounds[numSignals] =    { 1, 2, 2, 3, 2, 0, 2, 3, 2 };

/* Extra signals (for Team racing) when lights don't change, e.g. "Alert" short beeps etc.
 *   ASSUME SAME FOR ALL TEAM STARTS. +ve numbers are adjusted for each flight, as above
 */
int aTeamStartTimes[numSignals] = { -30, -20, -10, -5, 150, 160, 170, 175, NOTUSED }; // additional (sounds) in the last minute b4 each start
byte aTeamSounds[numSignals] =    {   2,   2,   2,  1,   2,   2,   2,   1, 0 }; // these are indexes into aSoundDefs


/* RELAY CONTROL: (Lamps) 
 * define the 3 LAMP relay actions (Class/Prep/Warn for each aStartTimes[] - ( Relay4 is the Horn, handled separately)
 * order is {Relay1,Relay2,Relay3}, where 0 = OFF, 1 = ON
 * So: 0 => All lamps OFF; 1 => Class Lamp On; 3 => Class + Prep Lamps ON;  4=> Warning Lamp (only) ON;
 * ISAF Sequence: 541GO { 0, 1, 3, 1, 0 }, 
 * With race warining at start: 10-541Go { 4, 1, 3, 1, 0 ), with flight repeat: { 4,1,3,1,0,1,3,1 0 }
 * (We ALWAYS specify including a flight repeat so we have all the timings)
 */
static byte aRelays[9] = { 4, 1, 3, 1, 0, 1, 3, 1, 0 }; // these are binary values using the bits b0..b2 to indicate lamp relays1-3 ON/OFF



/* **** BUTTONS: 
 *  Set the pins. Are active low with INPUT_PULLUP by default.
 */
RCS::Button startBtn(startBtnPin);    // pin 2
RCS::Button recallBtn(recallBtnPin);  // pin 8
RCS::Button hornBtn(hornBtnPin);      // pin 9
RCS::Button setBtn(setBtnPin);        // pin A5


/* ***** Initialise key (global) variables ****
 */
bool teamRace = false; // default
// if 1st char of sequence name is 'T' then teamRace will be set true during startup

byte sequenceID = 0; // the chosen sequence

int numFlights = 1; // default (max == 6) loaded from EEPROM, can change with config at startup

int flightNumber = 1; // Start with first flight, increments as each flight starts 


int rcstate = 0; // the state machine!
int numErrors = 0;

bool isCountdown = false;     // is set TRUE when START_btn is pressed
bool hornIsRunning = false;
bool msgFlag = false;         // flag we use to stop multiple outputs of same message to LCD or Serial Printing  
bool raceInProgress = false;  // after all flights started and no recalls, race is inProgress. We could enable a shorten course signal?  
  
const int buzzerSecs = 5;     // Warning buzzer sounds this secs before an action.

long int storeMillis;

// volatile vars becos we access from both ISR and main loop(0)
volatile bool showSecsCount = false;  
volatile long int timeCount = 0;     
volatile long int timeCountTen = 0;

// counters, for timing within main loop() method
long int loopCount;                   // a copy of the timeCount, update only at start of every loop
long int loopCountTen;                // ditto

bool bSetConfig = false;      // used only during "config"
bool bSetRepeat = false;      // used only during "config"





/****  METHODS (Functions) ****/

/* 
 * Interrupt Service Routine (ISR) for main race time counter (flexiTimer2)
 */
void flexiISR()
{
  static int LEDstate = LOW;  
  timeCountTen++;
  if (timeCountTen %10 == 0) {
    timeCount++;
    showSecsCount = true;  //for debugging
    LEDstate = !LEDstate;                 // gives a flashing LED at 1sec interval on pin 3 to 
    digitalWrite(stateLEDPin, LEDstate);  // just to show we are running - optional
  }
}



/* **** SETUP() ****
 * Allows the user to set the start sequence and number of flights (using the horn/recall buttons)
 *   sequenceID defines the start sequence required (default Rule 26, 541GO)
 *   numFlights defines how many flights (default = 1 i.e no repeats)
 */
void setup() {
    // runs once at power up
  bool bstartBtnWasPressed = false; // to detect start during config period    
  initHardware();

  if (debug) 
  {
    while (!Serial) { ; } // wait for serial port to connect. Needed for native USB port only
  }
  
  /* This program version has changed (compared to EEPROM stored value) */
  if ( ! Store.isCurrentVersion(versionNumber) ) 
  {
    Store.writeHeader(versionNumber); // update it
    Store.writeSequenceNumber(0);     // reset to first sequence ("541GO")
    Store.writeNumberOfFlights(1);    // and ONE flight
  }
 
  // load stored starting values from EEPROM
  sequenceID = Store.readSequenceNumber();
  numFlights = Store.readNumberOfFlights();

  if (haveLCD)
  {
    showSoftwareVersionStr();
    
    showStartSequenceName(sequenceID, 0);
    
    showNumberOfFlights(numFlights ,1);
    
    delay(1500); // pause to read
    
    // User can press Horn button within 15 secs to change settings
    // or press start to exit without changes
    if ( wantChangeSettings(15000) ) 
    {
      configureSettings(); // get user changes
    }
  }
  
  /* CONFIGURATION DONE (if user did any!) */
  
  // The Store object uses RCS_EE class (mine) to save/update values in EEPROM
  Store.writeSequenceNumber(sequenceID);  
  Store.writeNumberOfFlights(numFlights);
  
  loadStartSequenceArray(sequenceID);

  teamRace = ('T' == aSequenceNames[sequenceID][0]);
  
  // rcstate is our "state machine" variable
  rcstate = initRaceState(rcstate); // check action times are valid, sets rcstate to 1st valid.

  if ( numErrors == 0 )
  {
    getReady(); // Setup main race timer and give opening READY message  
  }
  else 
  {
    lcd.clear(); // lcd.setCursor(0,0);
    lcd.print(F("Errors=")); lcd.print(numErrors); 
    delay(10000); // 10 secs for user to read! then try to continue anyway.
  }

  // wait to start the race
  while (startBtn.onPressed() == false) { 
      checkHornButton(); // allowed before start to sound Postpone or Abandon
  } 

  // BEGIN!
  displayStartMessage();

  FlexiTimer2::start(); // start main (interrupt) timer.

  if (debug)  Serial.println("buzzer@ -5secs"); 
    
} // end of setup()




/* loop()
 *  
 *****   MAIN LOOP - FOR RUNNING THE RACE!  ****
 */
void loop() 
{
  // main code here, runs repeatedly:
  
  if (loopCountTen != timeCountTen) // timeCountTen is updated by the interrupt routine 
  {   // every 1/10th second
    loopCountTen = timeCountTen; // copy the value so it does not change during the below
    checkHornAndRecallButtons(); // for user presses
  }
  
  if (debug) 
   skipDeadTime(); // jump timeCount over quiet times, for speedy testing!

  if ( loopCount != timeCount ) // every second
  {  
    loopCount = timeCount; // new second, do stuff
    updateDisplays(loopCount);

    if (raceInProgress == false) 
    {  // we are in the start sequence
      
      if (flightNumber <= numFlights)
        doBuzzer(rcstate);  // check each second if warning buzzer should sound (5 secs before a command)      
    
      // The 'race control State Machine' will step through the states (shown by the rcstate variable)
      // This allows us to set some specific 'state' or timeCount, do some action ONCE
      //  then move on to the next 'state' so we can 'rinse & repeat'!
      
      if (rcstate >= 0 && rcstate != NOTUSED) // our valid states, zero-based
      {  
        // check horn and start a sound sequence if it is time.
        // NOTE: to allow for extra Team Racing sound signals need to check it every second, 
        // not just at main 'lights' action times 
        doHorn(); 

        if ( loopCount == aStartTimes[rcstate]) // action time!
        {            
          if (flightNumber <= numFlights) // if some flights are still to start
          {
            doRelays(rcstate); // turn the lights on/off as required

            // display message on the LCD screen e.g. "CLASS FLAG UP" etc.
            LCDdisplayTextLine(aStartNames[rcstate], 0);
            
            if (debug) {
              Serial.println(aStartNames[rcstate]); 
            }
          }
 
          // make any adjustements, then increment to next VALID state
          rcstate = checkStatus(rcstate);  
          // at T0 also updates flightNum, adjusts times/index for next flight
          // sets raceInProgress= true when all flights have started.
        
        } // endif (action time)
      }
    } // endif raceInProgress == false

  } // endif (loopCount != timeCount)
  
} // loop



/**** Functions / Methods ****/


/* for DEBUGGING ONLY 
 * skipDeadTime()
 * skips quiet bits to test more quickly!
 *  N.B changes the volatile variable used in the ISR!
 */
void skipDeadTime()
{
    if ( debug && abs(timeCount) > 50 )
      if ( ( timeCount < 0 ) && (abs(timeCount) % 60 == 55) )
        if ( !teamRace || ( timeCount < -100) ) // don't jump thru last bit is a team race, as need extra signals 
          timeCount = timeCount + 45;
}


/*
 * nextValidActionNdx(int state)
 * check array aStartTimes[state], skip any states that are NOTUSED
 * does NOT change rcstate global
 * RETURN (next) valid state
 */
int nextValidActionNdx(int state) {
  while ( aStartTimes[state] == NOTUSED)  { state++; }
  if (state >= aSTlength ) {
    if ( debug ) Serial.println(F("ERROR: nextValidAction overflow!"));
    doRaceIsInProgress(); // stops any more starting sequence actions
  }
  return state;
}


/*
 * doRaceIsInProgress()
 * When all flights have started, give messages
 */
void doRaceIsInProgress() {
  raceInProgress = true;
  if ( debug ) Serial.println(F("Race In Progress!"));
  LCDdisplayTextLine(F("Race IN PROGRESS"), 0); 
}


/* METHOD checkStatus(int rcstate)
 *  Called when the loopCount reaches an action time AND basic sounds and lights actions have been triggered
 *  at T0, increments flightNumber 
 *  at T0+1 sets raceInProgress=true if all flights have started 
 *  at last element of array:
 *    UpdateFlightTimes() Adds the flightInterval to all elements from T0 onwards, and resets rcstate to tZeroElement.
 *      (These are action times for the next flight start)
 *  RETURN: rcstate (array index) for the NEXT action
 */
int checkStatus(int rcstate) 
{
  // safety check  
  if (rcstate >= aSTlength ) {
    Serial.println(F("*** ERROR - rcstate index > aStartTimes[]")); numErrors++; return rcstate; }
    
  if (rcstate == aSTlength-1 )   // LAST entry in array.
  {
    updateFlightTimes();        // for the next flight
    rcstate = tZeroIndex;       // continue from this index for next flight 
  }

  if (rcstate == tZeroIndex) 
  { 
     flightNumber++;
     if ( debug) {  Serial.print(F("flight #"));Serial.println(flightNumber);
                   Serial.print(F("numFlights "));Serial.println(numFlights);
     }
  }
   
  if  (rcstate == tZeroIndex+1) // check if all done, accounts for any recalls
  {     // assume that if class recalled then numFlights++ in the btn method at tZero
     if ( flightNumber <= numFlights )  
       raceInProgress = false; // more flights starting
     else 
       doRaceIsInProgress();
  }
  
  // done the actions for this state, move to the next one
  rcstate++;    
  rcstate = nextValidActionNdx(rcstate); // skips any NOTUSED actions
  
  if ( debug) { // give debug message
     Serial.print(F("Next action [")); Serial.print(rcstate); Serial.print(F("] is at ")); Serial.println(aStartTimes[rcstate]);
  }
  return rcstate; 
}


/*  updateFlightTimes()
 *  Called when rcstate hits last item in aStartTimes, 
 *  Updates aStartTimes[] and aTeamStartTimes[] by adding the flight interval
 *   to each element from tZeroIndex (the flight GO time) onwards.
 */
void updateFlightTimes() {
    byte i;
  
  if (debug) { Serial.print(F("Next flight startTimes: ")); }

  for (i=tZeroIndex; i < aSTlength; i++ ) { // add flightInterval secs to each element from T0 onwards
    if ( aStartTimes[i] != NOTUSED )        // unless it's UNUSED magic number!
      aStartTimes[i] = aStartTimes[i] + flightInterval;
    if ( debug ) {  Serial.print(aStartTimes[i]); Serial.print(", "); }
  }
  if ( debug ) {  Serial.println(); }
  
  // if it is a team race, adjust the times for that too
  if (teamRace == true)
  {
    if ( debug ) {  Serial.print("Team Race signals: "); }
    i = 0;      
    while ( aTeamStartTimes[i] < 0 ) i++;
    while ( i < aSTlength )
    {
      if ( aTeamStartTimes[i] != NOTUSED )        // unless it's UNUSED magic number!
        aTeamStartTimes[i] = aTeamStartTimes[i] + flightInterval;
        if ( debug ) {  Serial.print(aTeamStartTimes[i]); Serial.print(", ");}
      i++;
    }
    if (debug) Serial.println();
  }  
}


/* getStartTimesMatch(bool isTeam)
 *  Uses global loopCount and start info arrays.
 * compare loopCount to the values in aStartTimes[]  
 * if none found and isTeam = true, then compare loopCount to the values in aTeamStartTimes[]
 * return the sndNdx required, or 0 if no match in either.
 * 
 * We use pointers to access the correct global arrays
 */
int getStartTimesMatch(bool isTeam) {
  byte i =0, aLen; byte *pSounds; // pointer to arrays (of bytes)
  int sndNdx = 0; int *pStarts;  // pointer to arrays (of ints) 
  
  // point to the main arrays and sounds
  pStarts = aStartTimes;
  aLen = aSTlength;
  pSounds = aSounds;
/* check pointers  
  result = *(pStarts);  // dereference to get the VALUE of the array element
  Serial.print("result[0] = ");Serial.println(result);
*/  
  if ( loopCount == NOTUSED) return 0; // no sounds allowed at this count!
  
  // compare each start time to loopCount, if found get the required sound index from sound array
  for (i=0; i < aLen; i++) {
    if ( *(pStarts+i) == loopCount) // dereference to get element value. 
      // get the matching aSoundDefs index from either aSounds[] or aTeamSounds[].
      sndNdx = *(pSounds+i);
  }  

  if ( (sndNdx == 0) && (isTeam ) ) // check extra team sounds
  {
    pStarts = aTeamStartTimes;
    aLen = arr_len(aTeamStartTimes);
    pSounds = aTeamSounds;

    for (i=0; i < aLen; i++) {
      if ( *(pStarts+i) == loopCount) // dereference to get element value. 
        // get the matching aSoundDefs index from either aSounds[] or aTeamSounds[].
        sndNdx = *(pSounds+i);
    }  
  }
  
  if ( debug && (sndNdx != 0) ) { 
    Serial.print(F("getStartTimesMatch sndNdx = "));Serial.println(sndNdx); }
  
  // aSoundDefs[sndNdx] is the sound sequence wanted by caller, or zero if none at this time
  return sndNdx;
      
}


/*
 * displayStartMessage()
 * On the LCD 
 */
void displayStartMessage() {
  if (haveLCD) {
    lcd.clear();
    // display action message on the LCD screen
    lcd.setCursor(0, 0);
    lcd.print(F("RUN "));
    lcd.setCursor(5, 0);
    lcd.print(aSequenceNames[sequenceID]);
    LCDshowCountDown(timeCount);    // shows '05' =  "buzzer" warning time
  }
}



// LCD methods

/*
 *  updateDisplays(long int counter)
 *  Display the race time on the LCD
 *   ( And counter on Serial for debugging )
 */
void updateDisplays(long int counter)
{
  if ( showSecsCount) {    // this is set true each second in the timer ISR
    if (debug) 
      Serial.println(counter,DEC); // print it once
    
    if (haveLCD) 
      LCDshowCountDown(counter);   // update LCD once
    
    showSecsCount = false; // then disable (until the next second)
  }
}


/* LCDdisplayTextLine(String text, int lineNum) 
 * Displays given text as a full line on LCD line 0 or 1
 */
void LCDdisplayTextLine(String text, int lineNum) 
{
  if ( haveLCD == false ) return;
  int l = (lineNum == 0 ?0 :1);
  lcd.setCursor(0,l);
  text += "                "; // append 16 spaces
  text.remove(16);            // cut to length 16 (i.e. a full line). 
  lcd.print(text); 
}


/*
 * LCDdisplayTextAt(String text, int x, int y) 
 */
void LCDdisplayTextAt(String text, int x, int y) 
{
  lcd.setCursor(x,y); // col, row
  lcd.print(text);
}


/* LCDshowCountDown(int loopSecs) 
 * Display time mmm:ss or -mm:ss
 * At 15,30 secs, show info messages
 */
void LCDshowCountDown(int loopSecs) 
{

  // LCD writing is slow, so only write minutes when it changes

  String m, s;
  byte minutes = byte(abs(loopSecs) / 60);
  byte seconds = abs(loopSecs) % 60;
  
  // format seconds to "ss"
  if (seconds < 10) 
    s = "0" + String(seconds); 
  else s = String(seconds); 
       
  lcd.setCursor(4,1); // always write the seconds
  lcd.print(s);
    
  // re-write when the minutes changes or at end of recall period (for poss flight addition)
  if (seconds == 0 || seconds == 59 )
  { 
    m = String(minutes);
    if (loopSecs < 0)  
      m = "-" + m; // set -ve minutes if we are counting down to race start
    while  (m.length() < 3) m = " " + m;  // pad mins

    lcd.setCursor(0,1); lcd.print(m); 
    lcd.setCursor(3,1); lcd.write(":");
  }

  // show additional info while we are here
  if (seconds == 15) 
  {
    lcd.setCursor(6,1);
    if (flightNumber <= numFlights) {
      lcd.print(" FLIGHT ");lcd.print(flightNumber);
    }else{
      lcd.print(" ELAPSED  ");
    }            
  }
  
  if (loopSecs == 30) 
  { // change race status at +30secs
    lcd.setCursor(0,0);
    lcd.print("RACE IN PROGRESS");
  }
}   // LCDshowCountDown


/* 
 *  *** doRelays((int rcstate)
 *  
 *  Drives lamp relays
 *  For given state, determine which relays should be ON or OFF
 *  for speed do not compare with previous lamp states, just do all 3! 
 *  Store the prevState so only do ONCE when state changes
 *  aRelays[rcstate] hold the 3 relays ON/OFF status for each rcstate (Class/Prep/Warn) in 1 byte. 
 *  bit b0=Class, b1=Prep, b2=Warn. b3 NOT USED (horn)
 *   example: ISAF sequence 541Go { 0, 1, 3, 1, 0,   1, 3, 1, 0}
 */
void doRelays(int rcstate) 
{
  byte seqID;
  
  // check valid rcstate?
  // get the lamp byte for this race state
  if ( rcstate >=0 && rcstate <= 9 ) // max of 9 race states for any lamp sequence
    seqID = aRelays[rcstate]; // e.g. aRelays[] = { 4,1,3,1,0, 1,3,1,0 }
  else
    seqID = 0; // all off

  if ( debug )
  {
    Serial.print(F("doRelays() "));
    if (seqID == 0) 
    {  Serial.print(F(" - all OFF"));
    }else{ 
       Serial.print(F("rcstate = "));Serial.print(rcstate);
    }
    Serial.println();
  }  
  // switch the relays ON or OFF according to bits 0..2
  digitalWrite(classRelayPin,seqID & 1?relayActiveState:!relayActiveState); // bit 0 ON is Class (Yellow)
  digitalWrite(prepRelayPin,seqID & 2?relayActiveState:!relayActiveState); // bit 1 ON is Prep (White)
  digitalWrite(warnRelayPin,seqID & 4?relayActiveState:!relayActiveState); // bit 2 ON is Race Warning (Blue)
}


/* 
 *  relayOnOff(int relayPin, int state) 
 */
void relayOnOff(int relayPin, int state) 
{
  if (relayActiveState)
    digitalWrite(relayPin,state);
  else   
    digitalWrite(relayPin,!state);
}

/*
 * relayToggle(int relayPin)
 * Flip the relay state
 */
void relayToggle(int relayPin)
{
  if (digitalRead(relayPin) == relayActiveState)
    digitalWrite(relayPin, !relayActiveState);
  else
    digitalWrite(relayPin, relayActiveState);
}


/* relayON(int relayPin) 
 */
void relayON(int relayPin) 
{
  digitalWrite(relayPin, relayActiveState); 
}

/* relayOFF(int relayPin) 
 */
void relayOFF(int relayPin) 
{
  digitalWrite(relayPin, !relayActiveState);
}


/* buzzerOn() 
 */
void buzzerOn() 
{
  pinMode(buzzerPin, OUTPUT);
  analogWrite(buzzerPin,128); // 50% duty cycle, default freq. 
}


/* buzzerOff() 
 */
void buzzerOff() 
{
  pinMode(buzzerPin, INPUT); // sound off
}


/*
 * doBuzzer( int rcstate)
 * toggle the buzzer On/Off in the 5 seconds before an action
 */
void doBuzzer( int rcstate) 
{
  static int prevCount = -999; // store for the counter value (after it has been actioned)
  int counter, i, diff;
  
  if (rcstate == NOTUSED) 
    return;

  counter = loopCount; // get timer count. NOTE: the count will be -ve before RACE START,+ve after
  if ( prevCount == counter ) // if already done this count, exit
    return;

  if (rcstate >=0 && rcstate < numSignals) // if valid state
  {  
    diff = aStartTimes[rcstate] - counter;
    
    if ( ( diff >= 0) && (diff <= 5) )  // if we are within 5 seconds of an action, toggle buzzer alternately On/Off
    {
      if ( odd(abs(diff)) ) 
        buzzerOn();
      else
        buzzerOff();
    } // if diff
  } // if rcstate
 
  prevCount = counter; // show the count has been actioned!
} // doBuzzer


/* UTILITY methods
 */

bool odd(int intVar) 
{
  return bool( intVar % 2 == 1);
}

bool even(int intVar) 
{
  return bool( intVar % 2 == 0);
}


