/* Race Control System - configuration methods
 *  Used at power up, to select the Start Sequence
 *  and number of flights (1..6)
 */

void showSoftwareVersionStr() {
  int i,k;
  char buf[8];
  for (i = 0;i<4; i++)
    buf[i] = Store.readByte(i);  
  // convert program version number to ascii chars, store in buf
  k = Store.readVersion();
  buf[4] = ( (k & 0xF0)/16) +48; // major version
  buf[5] = '.';
  buf[6] = (k & 0x0F) + 48;      // minor version
  buf[7] = '\0';

  if (haveLCD) {
    lcd.setCursor(8,1);
    lcd.print(buf);
    if ( debug ) Serial.println(buf);
  }
}

void showStartSequenceName(byte seqID, byte line) 
{
  if (haveLCD) 
    LCDdisplayTextLine(aSequenceNames[seqID], line);
}

void showNumberOfFlights(byte n, byte i) 
{
  if (haveLCD) {
    lcd.setCursor(0,i);
    lcd.print(F("       FLIGHTS="));lcd.print(n);
  }
}

bool wantChangeSettings(int checkForMSecs) 
{ 
    bool result = false;
    long int iMillis=0, endMillis;
    static int i=0;
  endMillis = millis() + checkForMSecs;   
  while ( !result && millis() < endMillis )
  {
    if (startBtn.isPressed() ) 
      endMillis = millis()-1; //exit now
    else
      result = hornBtn.onPressed();
    
    if (millis() > iMillis + 1000 )
    {
      iMillis = millis() +1000 ; 
      i++;
      if ( even(i) )
        showConfigMsgs(0); 
      else
        showNumberOfFlights(numFlights, 1);
    }
  }
  return result;
}

 
/* display 2 alternate messages on LCD line 1 */
void showConfigMsgs(byte i) 
{
  lcd.setCursor(0,1); 
  if ( i == 0 ) 
    lcd.print(F("(HORN TO CHANGE)"));
  else 
    lcd.print(F("(SET TO SAVE)   "));
}

/* User can set new values */
void configureSettings() 
{
  sequenceID = getNewSequence();
  numFlights = getNumFlights();
  showStartSequenceName(sequenceID, 0); //show (changed) values
  showNumberOfFlights(numFlights ,1);
}

byte getNewSequence() 
{
  showStartSequenceName(sequenceID,0); // current
  showConfigMsgs(1);
  while (! setBtn.onPressed() ) {
    if ( hornBtn.onPressed() ) {
      sequenceID = (sequenceID+1) % arr_len(aSequenceNames); // increment and wrap at end
      showStartSequenceName(sequenceID,0); 
      showConfigMsgs(1);
    }
  }
  return sequenceID;
}

byte getNumFlights() 
{
  showNumberOfFlights(numFlights, 0);
  showConfigMsgs(0);
  while (! setBtn.onPressed() ) {
    if ( hornBtn.onPressed() ) {
      if (++numFlights > 6 ) numFlights = 1; // increment and wrap
      showNumberOfFlights(numFlights, 0);
      showConfigMsgs(1);
    }    
  }
  return numFlights; 
}

/*
 * Initialise the pins, Serial and LCD
 */
void initHardware() {
  delay(100); // avoid spurious output on Serial etc. as it starts up

  if ( haveLCD == true ) {
      // set up the number of columns and rows on the LCD
      lcd.begin(16, 2);
      lcd.clear();
  }
  
  if (debug) {
    Serial.begin(57600); Serial.print("READY"); Serial.println();
  }
  
  // setup all pins for input/output
  pinMode(stateLEDPin, OUTPUT); // this pin will toggle on/off at 1 sec intervals
  digitalWrite(stateLEDPin,LOW); //switch it on to show we are alive and ready

  pinMode(startBtnPin, INPUT_PULLUP); // the start switch (momentary)

  pinMode(classRelayPin, OUTPUT);
  pinMode(prepRelayPin, OUTPUT);
  pinMode(warnRelayPin, OUTPUT);
  pinMode(hornRelayPin, OUTPUT);
    
  doRelays(-1); // all OFF 
 /*
  pinMode(4, OUTPUT); digitalWrite(4,LOW); // RELAY 4
  pinMode(5, OUTPUT); digitalWrite(5,LOW); // RELAY 3
  pinMode(6, OUTPUT); digitalWrite(6,LOW); // RELAY 2
  pinMode(7, OUTPUT); digitalWrite(7,LOW); // RELAY 1
*/
} // initHardware


/*  loadStartSequenceArray(int seqID) 
 *  load the required start sequence into RAM 
 *  need special function to read it back!
 */
void loadStartSequenceArray(int seqID) 
{  
  if (debug) { Serial.print(F("loading Start Sequence ")); Serial.println(seqID); }
  for ( int i = 0; i < numSignals; i++ ) 
  {
    aStartTimes[i] = pgm_read_word_near(aStartSeqs[seqID] + i ); // pgm_read_word_near(charSet + k);
    if (debug ) { Serial.print(aStartTimes[i]); Serial.print(", "); }
  }
  if (debug ) { Serial.println(); }
}


/* 
 * init some race global vars from the startTimes[] array 
 */
int initRaceState(int rcstate)
{  
  if ( debug ) Serial.print(F("initRaceState:"));

  aSTlength = arr_len(aStartTimes); // store array length to global

  // check action times are valid (i.e increasing sequence, with a zero element). store the aStartTimes[] tZeroIndex and aSTlength 
  if ( checkActionTimesValid() ) 
  {  
    // set the flightInterval = the LAST array entry
    flightInterval = aStartTimes[aSTlength - 1];

    // set state to FIRST valid action (index)
    rcstate = nextValidActionNdx(0); 
  }else
    numErrors++;
      
  if (debug ) { Serial.print(F(" T0 index = ")); Serial.print(tZeroIndex);
                Serial.print(F(", flightInterval = ")); Serial.print(flightInterval);
                if (rcstate != NOTUSED) { 
                  Serial.print(F(", nextActionTime = ")); Serial.println(aStartTimes[rcstate]); 
                }
              }   
  return rcstate;
}

/*
 * Check the LIGHTS action times are valid
 * return - index of FIRST valid action or 99 on error
 */
bool checkActionTimesValid() { 
  int el, val = -9999; // earlier than any actions  
  byte i = 0;
  
  // check elements are ascending or =NOTUSED, check and store index of T0 (zero) element
  tZeroIndex = NOTUSED;
  while ( i < aSTlength && numErrors == 0) 
  {
    el = aStartTimes[i];
    if  (el == 0) tZeroIndex = i; // race START element ( T0 ) found
         
    if (el != NOTUSED) 
      if ( el > val ) 
         val = el;      
      else
        numErrors++; // error count
    i++;   
  }

  // check results
  if ( (tZeroIndex == NOTUSED) || ( aStartTimes[aSTlength-1] == NOTUSED ) )
    numErrors++;
  if ( numErrors > 0 ) 
  { 
    if (debug) {Serial.print(F("ERROR: Invalid aStartTimes[] Start Sequence")); Serial.println(i);}
    if (haveLCD) 
    {
      lcd.clear();
      LCDdisplayTextLine(F("ERRORS in START"),0);
      lcd.setCursor(0,1);
      lcd.print(F("SEQUENCE: "));lcd.print(numErrors);
      delay(2000);
    }
  }
  return ( numErrors == 0);  
}


bool getReady() 
{
  timeCount = aStartTimes[rcstate]- buzzerSecs; // subtract the buzzer warning duration to get timeCount start value.

  // Initialise, but do not start Main Race Timer/Counter
  if ( ! debug ) {
    FlexiTimer2::set(100, flexiISR); // if LIVE, interrupt callback every 100 mSecs)
  }else{
    FlexiTimer2::set(100/speedUp, flexiISR); // for debugging we can speedUp the counter
  }
    
  if (debug) { Serial.print(F("timeCount= ")); Serial.println(timeCount); }
  if (haveLCD)
  {
    lcd.clear();
    lcd.print(F("READY  "));
    lcd.print(aSequenceNames[sequenceID]);
    showSoftwareVersionStr();
  }
  return numErrors == 0;
}

/*  ONLY for debugging, TO read ALL StartTimes data in PROGMEM & output to Serial */
void printAllStartTimesData() 
{  
  if (debug) {
    Serial.println(F("Starting Sequences are:"));
    int x = 0; int y = 0; int val = 0;
    for (y=0; y < 12; y++) {
      for ( x=0; x<9; x++) {
       // aStartTimes[x] = pgm_read_word_near(aStartSeqs[y] + x); //  array [y] acts as a dereference to the address of 1st dimension, so this works too!.
       val = pgm_read_word_near(&aStartSeqs[y][x]); // & is "the address of" operator, so we read the word at that address, assign to an int (rather than a word)
       Serial.print(x),Serial.print(":");Serial.print(val);Serial.print(", ");
      }
      Serial.println();
    }
  }
}

